<?php
declare(strict_types=1);

function installer_locked(): bool {
    return file_exists(__DIR__ . '/installer.lock');
}

function h(string $s): string {
    return htmlspecialchars($s, ENT_QUOTES, 'UTF-8');
}

function require_post_fields(array $fields): array {
    $data = [];
    foreach ($fields as $f) {
        $v = $_POST[$f] ?? '';
        if (!is_string($v)) $v = '';
        $v = trim($v);
        if ($v === '' && $f !== 'db_pass') {
            throw new RuntimeException("Missing field: {$f}");
        }
        $data[$f] = $v;
    }
    return $data;
}

function config_path(): string {
    return dirname(__DIR__) . '/config/config.php';
}

function ensure_config_dir(): void {
    $dir = dirname(config_path());
    if (!is_dir($dir)) {
        if (!mkdir($dir, 0755, true) && !is_dir($dir)) {
            throw new RuntimeException("Failed to create config directory: {$dir}");
        }
    }
}

function write_config(array $config): void {
    ensure_config_dir();
    $path = config_path();
    $content = "<?php\nreturn " . var_export($config, true) . ";\n";
    if (file_put_contents($path, $content, LOCK_EX) === false) {
        throw new RuntimeException("Failed to write config to {$path}. Check permissions.");
    }
}

function load_config(): array {
    $path = config_path();
    if (!file_exists($path)) throw new RuntimeException("Config not found. Run installer step 1.");
    $cfg = require $path;
    if (!is_array($cfg)) throw new RuntimeException("Invalid config format.");
    return $cfg;
}

function db_connect(array $cfg): mysqli {
    mysqli_report(MYSQLI_REPORT_OFF);
    $conn = @new mysqli($cfg['db_host'], $cfg['db_user'], $cfg['db_pass'], $cfg['db_name']);
    if ($conn->connect_error) {
        throw new RuntimeException("DB connection failed: " . $conn->connect_error);
    }
    $conn->set_charset('utf8mb4');
    return $conn;
}

function csrf_token(): string {
    if (session_status() !== PHP_SESSION_ACTIVE) session_start();
    if (empty($_SESSION['csrf'])) {
        $_SESSION['csrf'] = bin2hex(random_bytes(16));
    }
    return $_SESSION['csrf'];
}

function csrf_check(): void {
    if (session_status() !== PHP_SESSION_ACTIVE) session_start();
    $t = $_POST['csrf'] ?? '';
    if (!is_string($t) || empty($_SESSION['csrf']) || !hash_equals($_SESSION['csrf'], $t)) {
        throw new RuntimeException("Invalid CSRF token. Refresh the page and try again.");
    }
}
